/*****************************************************************************
| FILE:         inc_fi_base.cpp
| PROJECT:      G3g
| SW-COMPONENT: AmControllerPlugin
|------------------------------------------------------------------------------
| DESCRIPTION:  FI Messages over INC
|
|------------------------------------------------------------------------------
| COPYRIGHT:    (c) 2013 Robert Bosch GmbH
| HISTORY:
| Date      | Modification               | Author
| 03.06.13  | Initial revision           | Ranjit Katuri
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <sys/eventfd.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/time.h>
#include <time.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <assert.h>
#include <arpa/inet.h>
#include <sstream>
#include <algorithm>

#include <iostream>
#include "inc_fi_base.h"

using namespace std;

/**
* Static variables initialization
*/
static inc_fi_base* s_ptrself = NULL;
static bool bmsgpoolcreatebyme = false;
/**
* Lib globals
*/
DLT_DECLARE_CONTEXT(incfi_dltlog)

/**
* Function triggered on exit, safe case, if object is not destroyed by the application
*/
void inc_fi_base_on_exit()
{
  if(s_ptrself != NULL)
  {
    delete s_ptrself; // Trying to invoke destructor
    s_ptrself = NULL;
  }
}
/*****************************************************************************************************
*   INITIALIZATION FUNCTIONS
*****************************************************************************************************/
void inc_fi_init_dlt(DltContext& refcontext)
{
  incfi_dltlog = refcontext;
}

/*****************************************************************************************************
*   CLASSES FOR INTERNAL USE ONLY.
*****************************************************************************************************/

/*****************************************************************************************************
*   This class is to access the allocate functionality of amt_tclBaseMessage.
*****************************************************************************************************/
class HOLLOW_MAN amt_tcl_incmessage : public amt_tclMappableMessage
{
private:
  amt_tcl_incmessage();
  amt_tcl_incmessage(const amt_tcl_incmessage&);
  //Explicit private
  virtual amt_tclMappableMessage* poClone () const { return NULL;}
public:
  amt_tcl_incmessage(const OSAL_trMessage* pOSALHandle, tU32 u32MsgSize)
  :amt_tclMappableMessage(pOSALHandle,u32MsgSize)
  {
  }
  virtual ~amt_tcl_incmessage(){}
};

/*****************************************************************************************************
*   CLASS RELATED
*****************************************************************************************************/
/**
* Default Constructor
*/
inc_fi_base::inc_fi_base()
:m_sockfdccagw(EP_HANDLE_INVALID),
m_dgram(NULL),
m_mainloop(),
m_dptask(),
m_evfdccamsg(EP_HANDLE_INVALID),
m_ccamsglist()
{
  /**
  * Create message pool for OSAL Usage
  */
  if ( OSAL_s32MessagePoolCreate(300000)!= OSAL_OK )
  {
    //Failed to create a message pool... looks like someone has already created it..., just open it
    OSAL_s32MessagePoolOpen();
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_base::inc_fi_base Loaded OSAL Shared memory"));
  }
  else
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_base::inc_fi_base Created OSAL Shared memory"));
    //Message pool has been created by me...
    ::bmsgpoolcreatebyme = true;
  }
  /**
  * Initialize AMT Framework
  */
  amt_bInit();
  /**
  * Register on exit function for safety, in case if destructor is not invoked on this instance
  */
  atexit(inc_fi_base_on_exit);
  /**
  * Initialize the read buffer area
  */
  memset(m_rdbuffer,0,INC_CCA_MSG_MAX_LENGTH);
  /**
  * Creating Event fd for CCA Messages
  */
  m_evfdccamsg = eventfd(0,EFD_NONBLOCK);
  assert(m_evfdccamsg != EP_HANDLE_INVALID);//Validate fd
  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_base::inc_fi_base Fd Created for CCA Messages :"),DLT_INT(m_evfdccamsg));
  /**
  * Register Event FD With Mainloop
  */
  s_ptrself=this;

  /**
  * Start task to run main loop
  */
  if(m_mainloop.bAddfdtoPoll(m_evfdccamsg,POLLIN,inc_fi_base::vmainloopcb))
  {
    if(bSetupListener(INC_PORT_CCA_ADAPTER,INC_APPLICATION_HOST_NAME,INC_CCA_ADAPTER_HOST_NAME))
    {
      DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_base::inc_fi_base listener setup"));
      if(0 == pthread_create(&m_dptask,NULL,vDpTaskEntry,NULL))
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_base::inc_fi_base Setup Completed"));
      }
      else
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_base::inc_fi_base pthread_create failed "));
        assert(0);
      }
    }
    else
    {
      DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_base::inc_fi_base cca_inc_gateway is not running !!!! cannot setup listener "));
    }
  }
  else
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_base::inc_fi_base failed to add fd to poll"));
    assert(0);
  }
}
/**
* Destructor
*/
inc_fi_base::~inc_fi_base()
{
  m_mainloop.vStopMainLoop();//Stop mainloop
  s_ptrself = NULL; //Set pointer to NULL
  if(m_dgram != NULL)
  {
    dgram_exit(m_dgram);
    m_dgram = NULL;
  }
  close(m_sockfdccagw);
  if(::bmsgpoolcreatebyme)
  {
    //Deinitialize the message pool
    OSAL_s32MessagePoolDelete();
  }
  else
  {
    OSAL_s32MessagePoolClose();
  }

  amt_bDeInit();
}

/**
* Function to setup a listener for a given port on a given hostname
*/
bool inc_fi_base::bSetupListener(int inc_port, std::string /*lhostname*/, std::string rhostname)
{
  bool ret = false;
  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_base::bSetupListener, using addrinfo method "));
  struct addrinfo hints, *res;

  // first, load up address structs with getaddrinfo():
  memset(&hints, 0, sizeof hints);
  hints.ai_family = AF_INET;
  hints.ai_socktype = SOCK_STREAM;

  //Port number to String
  stringstream sport;
  sport << inc_port;

  if(0 == getaddrinfo(rhostname.c_str(), (sport.str()).c_str(), &hints, &res))
  {
    // make a socket:
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_base::bSetupListener, creating a socket "));
    m_sockfdccagw = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
    //m_sockfdccagw = socket(AF_INET,SOCK_STREAM,0);

    if(m_sockfdccagw > 0)
    {
      m_dgram = dgram_init(m_sockfdccagw,DGRAM_MAX,NULL);
      if(  connect(m_sockfdccagw, res->ai_addr, res->ai_addrlen) == 0)
      {
        //Do a dgram init now
        if(m_dgram != NULL)
        {
          //Set socket to nonblocking
          int on =1;
          int rc = ioctl(m_sockfdccagw, FIONBIO, (char *)&on);
          if (rc >= 0)
          {
            //Add this socket to main loop
            if(m_mainloop.bAddfdtoPoll(m_sockfdccagw,POLLIN,inc_fi_base::vmainloopcb))
            {
              ret = true;
            }
          }
          else
          {
            DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("Inc Setup: Failed to set socket to nonblocking"));
          }
        }
        else
        {
          DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("Inc Setup: DGRAM INIT failed"));
        }
      }
      else
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_base::bSetupListener, connect failed "));
      }
    }
    else
    {
      DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_base::bSetupListener, failed to create a socket "));
    }
  }
  if(!ret)
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("Inc Setup: Error : "),DLT_STRING(strerror(errno)));

    //Deinit dgram
    if(m_dgram != NULL)
    {
      dgram_exit(m_dgram); //Deintialize dgram here
      m_dgram = NULL;
    }

    //Close socket
    if(m_sockfdccagw != EP_HANDLE_INVALID)
    {
      close(m_sockfdccagw);
      m_sockfdccagw = EP_HANDLE_INVALID;
    }
  }

  return ret;
}

/**
* Entry function for mainloop task
*/
void* inc_fi_base::vDpTaskEntry(void* /*pvarg*/)
{
  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Inc Dispatcher task started "));
  if(s_ptrself != NULL)
  {
#if 0
    //Try setup listener
    uint32_t counter = 0;
    do
    {
      counter++;

      DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_base::vDpTaskEntry listener setup, attempt: "),DLT_UINT(counter));
      if(s_ptrself->bSetupListener(INC_PORT_CCA_ADAPTER,INC_APPLICATION_HOST_NAME,INC_CCA_ADAPTER_HOST_NAME))
        break;

      sleep(counter>5 ? 5 : counter);//Try with incremental sleeps till 5 seconds, and later a constant sleep
    }while(1);
#endif
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Inc Dispatcher: Listener setup done, Starting Mainloop"));
    s_ptrself->m_mainloop.vRunMainLoop();//Run continuosly
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Inc Dispatcher: Mainloop exit "));
  }
  else
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("Inc Dispatcher: ptrself is NULL !!! "));
  }

  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Inc Dispatcher task exited "));
  return NULL;
}
/**
* Callback receive when data is received
*/
void  inc_fi_base::vmainloopcb(int evFd, unsigned int events)
{
  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("INCFI EVENT, FD = "),DLT_INT(evFd),DLT_STRING(", events : "),DLT_UINT(events));

  if(s_ptrself != NULL)
  {
    /******************************************************************************
    *  EVENT HANDLER - INC SOCKETS
    ******************************************************************************/
    if(evFd == s_ptrself->m_sockfdccagw)//Check if FD belongs to INC Ports
    {
      assert(s_ptrself->m_dgram != NULL);

      int rc = EAGAIN;
      bool bmsgadded = false;

      do
      {
        //Now start reading
        rc= dgram_recv(s_ptrself->m_dgram,s_ptrself->m_rdbuffer,INC_CCA_MSG_MAX_LENGTH);

        if(rc > 0)
        {
          //Posting data
          DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("INCFI RX: "),DLT_RAW(s_ptrself->m_rdbuffer,rc));
        }
        else
          break;

        if(rc >= (int)AMT_C_U32_BASEMSG_ABSMSGSIZE) //expect some data atleast
        {
          //Create a osal message
          OSAL_trMessage rOSALMsg;
          if (OSAL_OK == OSAL_s32MessageCreate(&rOSALMsg,(uint32_t)rc , OSAL_EN_MEMORY_SHARED))
          {
            uint8_t* ptrData = OSAL_pu8MessageContentGet(rOSALMsg, OSAL_EN_READWRITE);
            if( 0 != ptrData)
            {
              OSAL_pvMemoryCopy(ptrData,(void*)s_ptrself->m_rdbuffer,rc);
            }
          }
          //Create a INC message with the message created
          amt_tcl_incmessage rxmsg(&rOSALMsg,(uint32_t)rc);

          //Now create a mappable message based on it
          amt_tclBaseMessage *ptrbasemsg = new amt_tclBaseMessage();

          //Handover contents to the message
          rxmsg.bHandOver(ptrbasemsg);

          //Message is ready, do dispatching
          s_ptrself->m_ccamsglist.push(ptrbasemsg);

          bmsgadded = true;
        }//else discard the message, dont need it
        else
        {
          DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Message discarded, invalid length"));
        }
      }while(rc > 0);//Read all datagrams, if queued

      if(bmsgadded)//Trigger event if you add messages to list
      {
        uint64_t data = 1;
        if(sizeof(uint64_t) != write(s_ptrself->m_evfdccamsg,&data,sizeof(data)))
        {
          DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Event post failed!! with error :"),DLT_STRING(strerror(errno)));
        }
      }
    }
    /******************************************************************************
    *  EVENT HANDLER - CCA MSG LIST
    ******************************************************************************/
    else if(s_ptrself->m_evfdccamsg == evFd)//Event handler for CCA Message
    {
      uint64_t data = 0;
      read(s_ptrself->m_evfdccamsg,&data,sizeof(data));//Clean the event field by triggering a read
      if(!s_ptrself->m_ccamsglist.empty())//There is some thing in the event
      {
        amt_tclBaseMessage* pmsg = s_ptrself->m_ccamsglist.front();//Get pointer
        s_ptrself->m_ccamsglist.pop();//Remove it from list
        s_ptrself->vHandleFiMsg(pmsg);//Process cca msg
        if(!s_ptrself->m_ccamsglist.empty())//Trigger event if there still some messages left in list
        {
          data = 1;
          write(s_ptrself->m_evfdccamsg,&data,sizeof(data));
        }
      }
    }
  }
}
/**
* CCA Message handler function
*/
void inc_fi_base::vHandleFiMsg(amt_tclBaseMessage* pmsg)
{
  if(pmsg != NULL)
  {
    switch(pmsg->u8GetType())
    {
    case AMT_C_U8_CCAMSGTYPE_SVCDATA:
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Msg Type: AMT_C_U8_CCAMSGTYPE_SVCDATA"));
        vOnNewMessage(pmsg);//Trigger vonnewmessage
      }
      break;
    case AMT_C_U8_CCAMSGTYPE_SVSSTATUS:
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Msg Type: AMT_C_U8_CCAMSGTYPE_SVSSTATUS"));

        uint16_t u16ServiceId, u16RegisterId, u16SourceAppId, u16SubId;
        uint8_t  u8ServiceState;
        // Downcast Object
        amt_tclServiceStatus oServiceStatus(pmsg);
        u16ServiceId = oServiceStatus.u16GetServiceID();
        u16RegisterId = oServiceStatus.u16GetRegisterID();
        u16SourceAppId = oServiceStatus.u16GetSourceAppID(); // server app id
        u8ServiceState = oServiceStatus.u8GetServiceState();
        u16SubId = oServiceStatus.u16GetTargetSubID(); // client sub id
        vOnServiceState( u16ServiceId, u16SourceAppId, u16RegisterId, u8ServiceState, u16SubId);
      }
      break;
    case AMT_C_U8_CCAMSGTYPE_SVCREGCONF:
    case AMT_C_U8_CCAMSGTYPE_POWER:
    case AMT_C_U8_CCAMSGTYPE_SVCREGISTER:
    case AMT_C_U8_CCAMSGTYPE_SVCUNREGISTER:
    case AMT_C_U8_CCAMSGTYPE_SUPPLIER_STATE:
    case AMT_C_U8_CCAMSGTYPE_APPLICATION_INFO:
    case AMT_C_U8_TYPE_DEBUG_APPLICATION:
    default:
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Message type is not handled"),DLT_UINT(pmsg->u8GetType()));
      }
      break;
    }
    pmsg->bDelete();//Free shared OSAL resources
    delete pmsg;//Free memory allocate for message
  }
}
/**
* Function to post a FI Message over INC
* Note> This function will consume the OSAL resources allocated for the message, but not the message itself
*/
bool inc_fi_base::bPostMessage (amt_tclBaseMessage* pomsg)
{
  bool ret = false;
  if(pomsg != NULL)
  {
    if((m_dgram != NULL)&&(pomsg->pu8GetSharedMemBase() != NULL))
    {
      //Set the source app ID as CCA Adapter
      pomsg->vSetSourceAppID(APP_ID_INC_CCA_ADAPTER);

      //This extraction is because, sometimes after writing, pomsg->u32GetSize() becomes zero.. dont know whyyy
      int32_t actsize = pomsg->u32GetSize();
      uint8_t* buffer = new uint8_t[actsize]();

      if(buffer != NULL)
      {
        memcpy(buffer,pomsg->pu8GetSharedMemBase(),actsize);

        DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("INCFI TX: "),DLT_RAW(buffer,actsize));

        //Send the message
        int rc = dgram_send(m_dgram, buffer,actsize);

        if(rc == actsize)
        {
          ret = true;
        }
        else
        {
          DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Failed to post message"));
        }
        delete[] buffer;
      }
      else
      {
        assert(0);//Out of memory !!!
      }
    }
    //Consume the memory allocated for the message
    pomsg->bDelete();
  }
  return ret;
}



